/**
*
* \file        dmci.c
*
* \brief       Support functions for DM input card operation when plugged into the DMCI
*              (card in a box ... ie standalone operation)
* \author      Larry Salant
* \date        3/17/2009
*
*/
#include <string.h>
#include <stdlib.h>
#include "dmci.h"
#include "leds.h"
#include "cnet_pnp.h"
#include "console.h"
#include "cresnet_slave.h"
#include "i2c.h"
#include "typedefs.h"
#ifdef ETHERNET_SUPPORTED
#include "ethernet.h"
#endif

////////////////////////////////////////////////////////////////////////////////
UINT8 CresnetLinkState;
INT16 CresnetLinkTimer;
UINT8 StandAloneMode = TRUE;
UINT8 gDMCIBus;
UINT8 (*pFunction_GetLinkStstus)(void) = 0;

#ifdef ETHERNET_SUPPORTED
UINT8 LinkState;
INT8 EthernetActivityTimer;
UINT8 EthernetActivityState;
#endif

// Definitions for I/O Expander
#define EXPANDER_ADDRESS 0x40  // on I2C
// i2c subaddresses / internal pca registers
#define PCA_INPUT_0                     0x00
#define PCA_INPUT_1                     0x01
#define PCA_OUTPUT_0                    0x02
#define PCA_OUTPUT_1                    0x03

#ifdef ETHERNET_SUPPORTED
    #ifdef STM32F2XX
        #include "stm32f2xx_eth.h"

        // Counters to keep track of ethernet packets for lighting up activity LED
        UINT32 g_EthHwRxCounter = 0, g_EthHwTxCounter = 0;
    #endif

/**
* \author    Larry Salant
* \brief     resets the timer for the Ethernet activity LED whenever we
*            get a new ethernet packet (used for the DMCI)
* \date      3/17/2009
* \param     none
* \return    void
* \retval    none
 */
void Ethernet_ActivityLED( void )
{
  EthernetActivityTimer = ETHERNET_ACTIVITY_STATUS_BLINK/DMCI_MONITOR_TASK_PERIOD;
}
#endif // ETHERNET_SUPPORTED

/**
* \author    Larry Salant
* \brief     Monitors DMCI switches and updates its LED's
*            called from periodic timer
* \date      3/17/2009
* \param     none
* \return    void
* \retval    none
**/
void DMCIMonitorTask(void)
{
  // check if the setup button is pressed.  Function also clears state, if necessary
  if (IsSetupButtonPressed())
  {
    // tell Cresnet the button was pressed
    PPNSetupButtonPressed();
  }

  // Update the Cresnet LED
  // the timer is set whenever we get polled on cresnet.
  // if the timer goes to 0, we haven't been polled in a while, turn off the LED, if it's not already off
  if( CresnetLinkTimer == 0 && CresnetLinkState)
  {
    // Make the LED off
    LedControl( LED_CRESNET, 0 );
    CresnetLinkState = 0;
  }
  // else, if there is time left
  else if (CresnetLinkTimer > 0)
  {
    // decrement the timer
    --CresnetLinkTimer;

    // if the LED is not already on,
    if (!CresnetLinkState)
    {
      LedControl( LED_CRESNET, 1 );//solid yellow on
      CresnetLinkState = 1;
    }
  }
#ifdef ETHERNET_SUPPORTED
  // For ethernet, we have to update 2 LED's:  Link and Activity
  // if we currently have link

  //if(pFunction_GetLinkStstus)
  // bCurrent = pFunction_GetLinkStstus();
  UINT8 bCurrent = DMCIGetLinkStatus();

  if (bCurrent)
  {
    // if link status changed, turn on the Link LED
    if (bCurrent != LinkState)
    {
      DMCILedControl(SLOT_ID_ETH_LINK_LED, ON);
      LinkState = bCurrent;
		ProcessGratuitousArp(GARP_LINK_UP);
    }
    
    #ifdef STM32F2XX
        // F2 devices have to poll the rx and tx counters to see if there's ethernet activity
        UINT32 ethHwRxCounter = ETH_GetMMCRegister(ETH_MMCRGUFCR);
        UINT32 ethHwTxCounter = ETH_GetMMCRegister(ETH_MMCTGFCR);
        
        if ((g_EthHwRxCounter != ethHwRxCounter) || (g_EthHwTxCounter != ethHwTxCounter))
        {
            Ethernet_ActivityLED(); // Light up activity LED
            g_EthHwRxCounter = ethHwRxCounter;
            g_EthHwTxCounter = ethHwTxCounter;
        }
    #endif

    // update the Activity LED
    // the timer is set whenever we get an ethernet packet
    // if the timer goes to 0, we haven't been polled in a while, turn off the LED, if it's not already off
    if( EthernetActivityTimer == 0 && EthernetActivityState)
    {
      // Turn the LED off
      DMCILedControl(SLOT_ID_ETH_ACT_LED, OFF);
      EthernetActivityState = 0;
    }
    // else, if there is time left
    else if (EthernetActivityTimer > 0)
    {
      // decrement the timer
      --EthernetActivityTimer;

      // if the LED is not already on,
      if (!EthernetActivityState)
      {
        DMCILedControl(SLOT_ID_ETH_ACT_LED, ON);
        EthernetActivityState = 1;
      }
    }
  }
  // no link now;  If we had it before, turn off both LED's
  else if (bCurrent != LinkState)
  {
    // turn off Link and Activity LED's
    DMCILedControl(SLOT_ID_ETH_LINK_LED, OFF);
    DMCILedControl(SLOT_ID_ETH_ACT_LED, OFF);
    LinkState = bCurrent;
  }

  //send gratuitous ARP when link up for static IP
    ProcessGratuitousArp(GARP_SEND_CHECK); 
#endif // ETHERNET_SUPPORTED
}
/**
* \author      Larry Salant
* \date        12/17/2008
* \brief       This routine will set the Cresnet ID of the system.
* \param       int ignore      - not used.
* \param       char *cmd       - string representing the address in dot notation.
* \return      int
* \retval      0 - successful command,  -1- if error occurred.
*/
INT32 CresnetIDCmd(UINT32 ignore, char *cmd)
{
  short CNetID;
  char *p ;
  char *endptr ;

  if (!StandAloneMode)
  {
    DmConsolePrintf("\tcannot set cresnet ID.\r");
    return -1;
  }

  if (cmd && *cmd == '?')         // print help strilng
  {
    DmConsolePrintf("CNETID [cresnet_id]\r");
    DmConsolePrintf("\tcresnet_id - ID of the cresnet node (in hex).\r");
    DmConsolePrintf("\tNo parameter - displays current setting\r");
    return 0;
  }
  if (!cmd || !(*cmd))                 // print current status
  {
    DmConsolePrintf("Cresnet ID = %x\r", CresnetGetDefaultId());
    return 0;
  }

  // decode argument string
  p = strtok(cmd," -.,;:") ;
  if ( !p )
    return -1;

  CNetID = (unsigned short)strtoul(p, &endptr, 16);  // read Cresnet_ID
  if ( (CNetID < MIN_DMCI_CRESNET_ADDRESS) || (CNetID > MAX_DMCI_CRESNET_ADDRESS) )
  {
    DmConsolePrintf("ERROR: Invalid Cresnet ID\r");
    return -1;
  }
  //Update NVL
  CresnetParamsSave(CNetID);

  DmConsolePrintf("New Cresnet ID = %2x.  Reboot to take effect.\r", CNetID);

  return 0;
}

/**
 * \author    Larry Salant
 * \brief     Set or Clears the specified LED on the DMCI when in standalone mode
 * \date      3/12/09
 * \param     Bus - I2C bus that the Port Expander with slot ID is on
 * \param     LED - which LED to change
 * \param     State - ON = turn on LED
 * \return    none
 * \retval    none
 * \note      ONLY used in standalone mode (otherwise, io expander configured as outputs
**/
void DMCILedControl(UINT8 bLED, UINT8 bState)
{
  UINT8 data;
  I2CADDR i2cAddr;

  // setup I2c addr of device, it's an 8 bit address, 8 bit subaddress
  i2cAddr.u.addr.type = I2C_ADDR_DEV8_SUB8;
  i2cAddr.u.dev8sub8.devAddr = EXPANDER_ADDRESS;

  // read the current device settings
  i2cAddr.u.dev8sub8.subAddr = PCA_INPUT_1;
  DmI2CReadBlock(gDMCIBus, (UINT32)i2cAddr.u.raw, &data, 1);

  // if LED is on
  if (bState)
    data |= bLED;  // set the mux to flip the bits
  else
    data = data & ~(bLED);  // set the mux to flip the bits

  // most write a word to the device to the device
  i2cAddr.u.dev8sub8.subAddr = PCA_OUTPUT_1;
  DmI2CWriteBlock(gDMCIBus, (UINT32)i2cAddr.u.raw, &data, 1);
}
